/**
  *******************************************************************************
  * 
  * @file    hal_gpio.c
  * @brief   ES7P203 gpio module driver.
  *
  * @version v1.0
  * @date 22 Aug 2018
  * @author  AE Team
  * @note
  *
  * copyright (C) shanghai Eastsoft Mictroelectornics Co. Ltd. All rights reseverd.
  *
  @verbatim
  =================================================================================
  			##### How to use this driver #####
  =================================================================================
  [..]  The Gpio driver can be used as follows:
    (+) Initialize the gpio registers by using gpio_init().
    (+) Set gpio pin out value by using gpio_write_pin()
    (+) Read gpio input value by using gpio_read_pin()

  @endverbatim
  *********************************************************************************
  */
#include "hal_gpio.h"

/**
  * @addtogroup ES7P203_HAL
  * @{
  */

/**
  * @defgroup gpio base
  * @brief gpio module driver
  * @{
  */

/**
  * @defgroup Gpio_Public_Functions gpio Public Functions
  * @brief    Gpio Functions
@verbatim
=================================================================================
		##### Flash Public Functions #####
=================================================================================
  [..]  The gpio driver can be used as follows:
    (+) Initialize the gpio registers by using gpio_init().
    (+) Set gpio pin out value by using gpio_write_pin()
    (+) Read gpio input value by using gpio_read_pin()
@endverbatim
  * @{
  */

/**
  * @brief  Gpio init
  * @param  gpio: The type of gpio
  * @param  pin: The pin number of gpio
  * @param  init: The point of init struct
  * @retval Status, see @ref hal_status_t.
  */
hal_status_t gpio_init(gpio_type_t gpio, gpio_pin_t pin, gpio_init_t *init)
{
	if (init == NULL)
		return ERROR;
	
	assert_param(IS_GPIO_TYPE(gpio));
	assert_param(IS_GPIO_IO_TYPE(init->io));
	assert_param(IS_GPIO_LEVEL_TYPE(init->level));
	assert_param(IS_GPIO_MODE_TYPE(init->mode));
	assert_param(IS_GPIO_DRIVER_TYPE(init->drive));
	assert_param(IS_GPIO_PIN_TYPE(pin));
	
	switch (gpio) {
	case PA_TYPE:
		PAT  &= ~(PAT & ((~init->io & 0x1) << pin)); 
		PA   &= ~(PA & ((~init->level & 0x1) << pin));
		PAS  &= ~(PAS & ((~init->mode & 0x1) << pin));
		PAPU |= init->pu << pin;
		PAPD |= init->pd << pin;
		PALC |= init->drive << pin;
		PAOD |= init->od << pin;
		
		if (init->cur_ctr == ENABLE) {
			assert_param(IS_GPIO_SC_TYPE(init->src_cur));
			if (pin <= GPIO_PIN_3) {
				SCC    |= init->cur_ctr << 6;
				SCSEL0 |= init->src_cur << 4; 
			}
			else {
				SCC    |= init->cur_ctr << 7;
				SCSEL0 |= init->src_cur << 6;
			}
		}
		break;
	case PB_TYPE:
		PBT    &= ~(PBT & ((~init->io & 0x1) << pin)); 
		PB     &= ~(PB & ((~init->level & 0x1) << pin));
		PBS    &= ~(PBS & ((~init->mode & 0x1) << pin));
		PBPU   |= init->pu << pin;
		PBPD   |= init->pd << pin;
		PBLC   |= init->drive << pin;
		PBOD   |= init->od << pin;
		PBSMIT |= init->smit << pin;
		
		if (init->cur_ctr == ENABLE) {
			assert_param(IS_GPIO_SC_TYPE(init->src_cur));
			if (pin <= GPIO_PIN_3) {
				SCC    |= init->cur_ctr << 4;
				SCSEL0 |= init->src_cur << 0; 
			}
			else {
				SCC    |= init->cur_ctr << 5;
				SCSEL0 |= init->src_cur << 2;
			}
		}
		break;
	case PE_TYPE:
		PET    &= ~(PET & ((~init->io & 0x1) << pin)); 
		PE     &= ~(PE & ((~init->level & 0x1) << pin));
		PES    &= ~(PES & ((~init->mode & 0x1) << pin));
		PEPU   |= init->pu << pin;
		PEPD   |= init->pd << pin;
		PELC   |= init->drive << pin;
		PEOD   |= init->od << pin;
		PESMIT |= init->smit << pin;
		
		if (init->cur_ctr == ENABLE) {
			assert_param(IS_GPIO_SC_TYPE(init->src_cur));
			if (pin <= GPIO_PIN_3) {
				SCC    |= init->cur_ctr << 0;
				SCSEL1 |= init->src_cur << 0; 
			}
			else {
				SCC    |= init->cur_ctr << 1;
				SCSEL1 |= init->src_cur << 2;
			}
		}
	default:
		break;
	}
	
	return OK;
}

/**
  * @brief  Set gpio pin out value
  * @param  gpio: The type of gpio
  * @param  pin: The pin number of gpio
  * @param  val: The value of gpio output
  * @retval None.
  */
void gpio_write_pin(gpio_type_t gpio, gpio_pin_t pin, uint8_t val)
{
	assert_param(IS_GPIO_TYPE(gpio));
	assert_param(IS_GPIO_PIN_TYPE(pin));
	
	switch (gpio) {
	case PA_TYPE:
		PA = val << pin;
		break;
	case PB_TYPE:
		PB = val << pin;
		break;
	case PE_TYPE:
		PE = val << pin;
		break;
	default:
		break;
	}
	
	return;
}

/**
  * @brief  Read gpio input value
  * @param  gpio: The type of gpio
  * @param  pin: The pin number of gpio
  * @retval The value of read.
  */
uint8_t gpio_read_pin(gpio_type_t gpio, gpio_pin_t pin)
{
	assert_param(IS_GPIO_TYPE(gpio));
	assert_param(IS_GPIO_PIN_TYPE(pin));
	
	switch (gpio) {
	case PA_TYPE:
		return ((PA >> pin) & 0x1);
	case PB_TYPE:
		return ((PB >> pin) & 0x1);
	case PE_TYPE:
		return ((PE >> pin) & 0x1);
	default:
		return 0;
	}
}

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */
